<?php
/**
 * consultar_avaliacao.php - Painel de consulta de avaliações para o Funcionário
 * Este arquivo permite que qualquer funcionário logado veja suas próprias avaliações finalizadas.
 * Permite a seleção e visualização de avaliações passadas.
 *
 * Requisitos:
 * - PHP 7.4+
 * - MySQL/MariaDB (mysqli)
 * - conexao.php com $conn (mysqli)
 * - Sessão com $_SESSION['usuario']['perfil'] e ['id']
 */

session_start();
require_once __DIR__ . '/conexao.php';

// ======= CONTROLE DE ACESSO =======
$perfil = strtoupper((string)($_SESSION['usuario']['perfil'] ?? ''));
$usuario_id = $_SESSION['usuario']['id'] ?? null;
$nomeUsuario = (string)($_SESSION['usuario']['nome'] ?? '');

// ✅ Apenas verifique se há um usuário logado
if ($usuario_id === null) {
    header('HTTP/1.1 403 Forbidden');
    echo '<h2>Acesso negado</h2><p>Você precisa estar logado para acessar este painel.</p>';
    exit;
}

// ======= CSRF (para manter a consistência) =======
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf = $_SESSION['csrf_token'];

// ======= FUNÇÕES AUXILIARES =======
function all($conn, $sql, $types = null, ...$params) {
    if ($types) {
        $stmt = $conn->prepare($sql);
        if ($stmt === false) { return []; }
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $res = $stmt->get_result();
        $stmt->close();
    } else {
        $res = $conn->query($sql);
    }
    $o = [];
    if ($res) {
        while ($r = $res->fetch_assoc()) {
            $o[] = $r;
        }
    }
    return $o;
}
function h($s) {
    return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8');
}
function getUserNameById($conn, $id) {
    $stmt = $conn->prepare("SELECT nome FROM usuarios WHERE id = ?");
    $stmt->bind_param('i', $id);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $stmt->close();
    return $user ? $user['nome'] : 'N/A';
}

// Carregar critérios de avaliação para exibição
$criterios = all($conn, "SELECT * FROM criterios_avaliacao WHERE ativo=1 ORDER BY categoria, ordem, nome");

// ======= CARREGAR AVALIAÇÃO SELECIONADA =======
$avaliacao = null;
$detalhes = [];
$nomeAvaliador = 'N/A';
$idSelecionado = (isset($_GET['id']) && ctype_digit($_GET['id'])) ? (int)$_GET['id'] : 0;

if ($idSelecionado > 0) {
    // Importante: Consulta avalia apenas a avaliação do usuário logado e que esteja FINALIZADA
    $st = $conn->prepare("SELECT * FROM avaliacoes WHERE id=? AND usuario_id=? AND estado='FINAL'");
    $st->bind_param('ii', $idSelecionado, $usuario_id);
    $st->execute();
    $avaliacao = $st->get_result()->fetch_assoc();
    $st->close();
    if ($avaliacao) {
        $rows = all($conn, "SELECT * FROM avaliacao_detalhes WHERE avaliacao_id=" . (int)$avaliacao['id']);
        foreach ($rows as $r) {
            $detalhes[(int)$r['criterio_id']] = $r;
        }
        $nomeAvaliador = getUserNameById($conn, $avaliacao['avaliador_id']);
    }
}

// ======= HISTÓRICO DAS AVALIAÇÕES FINALIZADAS DO PROFESSOR LOGADO =======
$historico = all($conn,
    "SELECT * FROM avaliacoes WHERE usuario_id = ? AND estado = 'FINAL' ORDER BY created_at DESC",
    'i', $usuario_id
);

// Se nenhuma avaliação foi selecionada, e houver histórico, exibe a mais recente
if (!$idSelecionado && !empty($historico)) {
    $idSelecionado = (int)$historico[0]['id'];
    $avaliacao = $historico[0];
    $rows = all($conn, "SELECT * FROM avaliacao_detalhes WHERE avaliacao_id=" . $idSelecionado);
    foreach ($rows as $r) {
        $detalhes[(int)$r['criterio_id']] = $r;
    }
    $nomeAvaliador = getUserNameById($conn, $avaliacao['avaliador_id']);
}
?>
<!DOCTYPE html>
<html lang="pt">
<head>
<meta charset="utf-8" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
<title>Minhas Avaliações — SisMAG</title>
<style>
/* CSS do tema escuro original */
:root{--bg:#0b1220;--panel:#0f172a;--card:#111827;--muted:#94a3b8;--text:#e5e7eb;--pri:#2563eb;--pri2:#1d4ed8;--ok:#16a34a;--warn:#d97706;--err:#dc2626}
*{box-sizing:border-box}
body{margin:0;background:linear-gradient(180deg,var(--bg),#0a1020);color:var(--text);font-family:Inter,system-ui,Segoe UI,Arial,sans-serif}
.header{position:sticky;top:0;background:rgba(15,23,42,.85);backdrop-filter:blur(8px);border-bottom:1px solid #1f2937;z-index:10}
.container{max-width:1200px;margin:0 auto;padding:16px}
.hflex{display:flex;align-items:center;gap:12px}
.logo{font-weight:800}
.badge{background:var(--pri);color:#fff;border-radius:999px;padding:6px 10px;font-size:12px}
.grid{display:grid;gap:16px}
@media(min-width:900px){.grid{grid-template-columns:1.2fr .8fr}}
.card{background:linear-gradient(180deg,rgba(17,24,39,.9),rgba(17,24,39,.7));border:1px solid #1f2937;border-radius:18px;padding:16px;box-shadow:0 10px 25px rgba(0,0,0,.25)}
.label{font-size:12px;color:var(--muted);margin-bottom:6px}
.input,select,textarea{width:100%;background:#0b1220;border:1px solid #253049;color:#dbeafe;border-radius:12px;padding:10px 12px;outline:none}
.row{display:grid;grid-template-columns:1fr 1fr;gap:12px}
@media(max-width:720px){.row{grid-template-columns:1fr}}
.kpi{display:flex;gap:10px;flex-wrap:wrap}
.pill{background:#0b1220;border:1px solid #253049;border-radius:12px;padding:8px 12px}
.btns{display:flex;gap:8px;flex-wrap:wrap}
.btn{background:var(--pri);border:none;color:#fff;padding:10px 14px;border-radius:12px;cursor:pointer;font-weight:700}
.btn:hover{background:var(--pri2)}
.btn.outline{background:transparent;border:1px solid #334155;color:#cbd5e1}
.btn.ok{background:var(--ok)}
.btn.warn{background:var(--warn)}
.btn.err{background:var(--err)}
.table{width:100%;border-collapse:collapse}
.table th,.table td{padding:10px;border-bottom:1px solid #253049;text-align:left}
.table th{color:#93c5fd}
.state{padding:2px 8px;border-radius:999px;font-size:12px;display:inline-block}
.state.RASCUNHO{background:#1f2937;color:#fbbf24;border:1px solid #92400e}
.state.FINAL{background:#052e16;color:#86efac;border:1px solid #166534}
.score{font-size:28px;font-weight:800}
.crit{display:grid;grid-template-columns:1.2fr .6fr;gap:10px;align-items:center;border:1px solid #253049;border-radius:14px;padding:12px;margin-bottom:10px}
.slider{width:100%}
.small{font-size:12px;color:#a5b4fc}
.print-hide{display:block}
.print-only{display:none}

/* CSS para Impressão */
@media print {
    .print-hide{display:none}
    .print-only{display:block;color:#000;background:#fff;padding:0;margin:0}
    body{background:#fff;color:#000;padding:0;margin:0;font-family:serif}
    .header,.card,aside,.logo,.badge{box-shadow:none;border:none}
    .print-grid{display:grid;gap:20px;grid-template-columns:1fr}
    h1,h2,h3{page-break-after:avoid}
    .print-signatures{display:flex;justify-content:space-around;margin-top:40px;text-align:center}
    .print-signatures div{width:45%;padding-top:20px;border-top:1px solid #000}
    .print-table{width:100%;border-collapse:collapse;margin-top:15px}
    .print-table th, .print-table td{padding:8px;border:1px solid #ccc;text-align:left;font-size:14px}
    .print-table th{background:#f0f0f0;font-weight:bold}
    .print-row{border-bottom:1px solid #e0e0e0;padding:10px 0}
}
</style>
</head>
<body>
<header class="header print-hide">
    <div class="container hflex">
        <div class="logo">SISMAG · Minhas Avaliações</div>
        <span class="badge"><?= h($perfil) ?></span>
        <div style="margin-left:auto" class="small">Utilizador: <?= h($nomeUsuario) ?></div>
    </div>
</header>

<div class="container grid print-hide">
    <section class="card">
        <h3>Detalhes da Avaliação</h3>
        <?php if ($avaliacao): ?>
            <div class="kpi" style="margin-bottom:10px">
                <div class="pill">Período: <span style="font-weight:700"><?= h($avaliacao['periodo']) ?></span></div>
                <div class="pill">Data: <span style="font-weight:700"><?= h($avaliacao['data_avaliacao']) ?></span></div>
                <div class="pill">Avaliador: <span style="font-weight:700"><?= h($nomeAvaliador) ?></span></div>
            </div>
            <div class="kpi">
                <div class="pill">Pontuação total (0–20): <span id="score" class="score"><?= number_format((float)$avaliacao['total_20'], 2) ?></span></div>
                <button class="btn warn" onclick="window.print()">🖨 Imprimir</button>
            </div>

            <hr style="border-color:#253049;border-width:1px 0 0;margin:14px 0;" />

            <?php
            $catAtual = '';
            foreach ($criterios as $c):
                if ($catAtual !== $c['categoria']) {
                    if ($catAtual !== '') echo '<br />';
                    echo '<h3>' . h($c['categoria']) . '</h3>';
                    $catAtual = $c['categoria'];
                }
                $cid = (int)$c['id'];
                $notaPre = isset($detalhes[$cid]) ? (float)$detalhes[$cid]['nota5'] : 0.0;
                $comentPre = isset($detalhes[$cid]) ? (string)$detalhes[$cid]['comentario'] : 'N/A';
            ?>
                <div class="crit">
                    <div>
                        <div style="font-weight:700"><?= h($c['nome']) ?> <span class="small">(peso <?= h($c['peso']) ?>)</span></div>
                        <div class="small"><?= h($c['descricao']) ?></div>
                        <p class="small" style="margin:5px 0 0;">Comentário: <?= h($comentPre) ?></p>
                    </div>
                    <div>
                        <div class="pill" style="margin-top:8px"><span class="small">Nota:</span> <strong class="nota-val"><?= number_format($notaPre, 1) ?></strong> / 5</div>
                    </div>
                </div>
            <?php endforeach; ?>

            <div class="row">
                <div>
                    <div class="label">Observações gerais</div>
                    <p><?= nl2br(h($avaliacao['observacoes'])) ?></p>
                </div>
                <div>
                    <div class="label">Plano de melhoria</div>
                    <p><?= nl2br(h($avaliacao['plano_melhoria'])) ?></p>
                </div>
            </div>
        <?php else: ?>
            <p>Selecione uma avaliação na lista ao lado para ver os detalhes, ou verifique se ela já foi finalizada pela Direção.</p>
        <?php endif; ?>
    </section>

    <aside class="card">
        <h3>Minhas Avaliações Finalizadas</h3>
        <?php if ($historico): ?>
            <table class="table">
                <thead>
                    <tr>
                        <th>Período</th>
                        <th>Data</th>
                        <th>Total</th>
                        <th></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($historico as $h): ?>
                        <tr>
                            <td><?= h($h['periodo']) ?></td>
                            <td><?= h($h['data_avaliacao']) ?></td>
                            <td><?= number_format((float)$h['total_20'], 2) ?></td>
                            <td><a class="btn outline" href="?id=<?= (int)$h['id'] ?>">Ver</a></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <p>Nenhuma avaliação finalizada encontrada para você.</p>
        <?php endif; ?>
    </aside>
</div>

<div class="print-only">
    <div style="text-align:center;margin-bottom:20px;padding-top:20px;">
        <h2>Avaliação de Desempenho do Funcionário</h2>
        <p><strong>Funcionário:</strong> <?= h($nomeUsuario) ?></p>
        <p><strong>Período:</strong> <?= h($avaliacao['periodo'] ?? 'N/A') ?></p>
        <p><strong>Data da Avaliação:</strong> <?= h($avaliacao['data_avaliacao'] ?? 'N/A') ?></p>
        <p><strong>Avaliador:</strong> <?= h($nomeAvaliador) ?></p>
    </div>

    <div style="padding:0 20px;">
        <h3>Pontuação Final: <span style="font-size:24px;color:green;"><?= number_format((float)($avaliacao['total_20'] ?? 0), 2) ?></span> / 20</h3>

        <table class="print-table">
            <thead>
                <tr>
                    <th>Critério de Avaliação</th>
                    <th>Peso</th>
                    <th>Nota (0-5)</th>
                    <th>Comentário</th>
                </tr>
            </thead>
            <tbody>
                <?php
                $catAtual = '';
                foreach ($criterios as $c):
                    if ($catAtual !== $c['categoria']) {
                        echo '<tr><td colspan="4" style="font-weight:bold;background:#f0f0f0;">' . h($c['categoria']) . '</td></tr>';
                        $catAtual = $c['categoria'];
                    }
                    $cid = (int)$c['id'];
                    $notaPre = isset($detalhes[$cid]) ? (float)$detalhes[$cid]['nota5'] : 0.0;
                    $comentPre = isset($detalhes[$cid]) ? (string)$detalhes[$cid]['comentario'] : 'N/A';
                ?>
                <tr>
                    <td><?= h($c['nome']) ?></td>
                    <td><?= number_format((float)$c['peso'], 2) ?></td>
                    <td><?= number_format($notaPre, 1) ?></td>
                    <td><?= h($comentPre) ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>

        <br>
        <p><strong>Observações Gerais:</strong><br><?= nl2br(h($avaliacao['observacoes'] ?? 'N/A')) ?></p>
        <p><strong>Plano de Melhoria:</strong><br><?= nl2br(h($avaliacao['plano_melhoria'] ?? 'N/A')) ?></p>
    </div>

    <div class="print-signatures">
        <div>
            <p>_____________________________________</p>
            <p>Assinatura do Funcionário Avaliado</p>
            <p>Nome: <?= h($nomeUsuario) ?></p>
        </div>
        <div>
            <p>_____________________________________</p>
            <p>Assinatura do Avaliador</p>
            <p>Nome: <?= h($nomeAvaliador) ?></p>
        </div>
    </div>
</div>
</body>
</html>